package server;

import javax.swing.JOptionPane;
import java.rmi.RMISecurityManager;
import gui.*;

public class Main {
	
	final private static String helpMessage = "Executes the Connect Four Matchmaking Server with the specified input parameters.\n\n" +
	  "usage: java server.Main [-q \"Server Queue Name\"] [-rmiport 1099] [-b \"127.0.0.1\"]\n\n" +
	  " -q :: specifies the name of the server's queue; e.g., -q \"Matchmaking Server Queue\"\n" +
	  "       (if no port is specified, defaults to \"Matchmaking Server Queue\")\n" +
	  " -rmiport :: specifies which port to use for rmi; e.g., -rmiport 1099\n" +
	  "             (if no port is specified, defaults to 1099)\n" +
	  " -b :: specifies the host name of the broker; e.g., -b \"127.0.0.1\"\n" +
	  "       (if no host name is specified, default is used)";
	
	public static void main(String[] args) {
		// pre-configure the policy file and install the security manager
		System.setProperty("java.security.policy", "all.policy");
		System.setSecurityManager(new RMISecurityManager());
		
		// parse input parameters
		parseParameters(args);
	}
	
	public static void displayHelp() {
		// print instructions on how to use program parameters
		JOptionPane.showMessageDialog(null, helpMessage, "Help", JOptionPane.INFORMATION_MESSAGE);
	}
	
	public static void parseParameters(String[] args) {
		// verify that there is a correct number of arguments
		if(args != null && args.length % 2 != 0) {
			displayHelp();
			return;
		}
		
		int registryPort = 1099;
		String serverQueueName = null;
		String brokerHostName = null;
		
		// parse program arguments
		if(args != null) {
			for(int i=0;i<args.length;i+=2) {
				// parse server queue name
				if(args[i].equalsIgnoreCase("-q")) {
					String temp = args[i+1];
					if(temp == null) {
						JOptionPane.showMessageDialog(null, "Server queue name cannot be null.", "Invalid Argument", JOptionPane.ERROR_MESSAGE);
						displayHelp();
						return;
					}
					serverQueueName = temp.trim();
					if(serverQueueName.length() == 0) {
						JOptionPane.showMessageDialog(null, "Server queue name cannot be empty.", "Invalid Argument", JOptionPane.ERROR_MESSAGE);
						displayHelp();
						return;
					}
				}
				// parse rmi port
				else if(args[i].equalsIgnoreCase("-rmiport")) {
					String temp = args[i+1];
					if(temp == null) {
						JOptionPane.showMessageDialog(null, "Registry port cannot be null.", "Invalid Argument", JOptionPane.ERROR_MESSAGE);
						displayHelp();
						return;
					}
					int port = -1;
					try { port = Integer.parseInt(temp); }
					catch(NumberFormatException e) { }
					if(port < 1 || port > 65535) {
						JOptionPane.showMessageDialog(null, "Registry port must be between 1 and 65535.", "Invalid Argument", JOptionPane.ERROR_MESSAGE);
						displayHelp();
						return;
					}
					registryPort = port;
				}
				// parse broker host name
				else if(args[i].equalsIgnoreCase("-b")) {
					String temp = args[i+1];
					if(temp == null) {
						JOptionPane.showMessageDialog(null, "Broker host name cannot be null.", "Invalid Argument", JOptionPane.ERROR_MESSAGE);
						displayHelp();
						return;
					}
					brokerHostName = temp.trim();
					if(brokerHostName.length() == 0) {
						JOptionPane.showMessageDialog(null, "Broker host name cannot be empty.", "Invalid Argument", JOptionPane.ERROR_MESSAGE);
						displayHelp();
						return;
					}
				}
				// handle invalid arguments
				else {
					JOptionPane.showMessageDialog(null, "Invalid argument: " + args[i], "Invalid Argument", JOptionPane.ERROR_MESSAGE);
					displayHelp();
					return;
				}
			}
		}
		
		// initialize the dbms window
		ServerWindow window = new ServerWindow();
		window.initialize(serverQueueName, registryPort, brokerHostName);
	}
	
}
